import {
  createUserWithEmailAndPassword,
  getAdditionalUserInfo,
  GoogleAuthProvider,
  onAuthStateChanged,
  sendEmailVerification,
  sendPasswordResetEmail,
  signInWithEmailAndPassword,
  signInWithPopup,
  signOut,
  updateProfile,
} from "firebase/auth";
import { useEffect, useState } from "react";
import { firebaseAuth } from "@/lib/firebase";
import { createSession, deleteSession } from "@/actions/auth-actions";
import { useCreateUser } from "@/services/user-services";

// Format the user object
function formatAuthUser(user) {
  return user
    ? {
        uid: user?.uid,
        email: user?.email,
        displayName: user?.displayName,
        photoURL: user?.photoURL,
      }
    : null;
}

// Custom hook to handle Firebase authentication
export default function useFirebaseAuth() {
  const [authUser, setAuthUser] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const {
    trigger: createUser,
    error: createUserError,
    isMutating,
  } = useCreateUser();

  async function authStateChanged(authState) {
    try {
      if (authState) {
        setAuthUser(formatAuthUser(authState));
      } else {
        setAuthUser(null);
      }
    } catch (error) {
      console.log(error);
    } finally {
      setLoading(false);
    }
  }

  // Create a new user with email and password
  async function createUserWithEmailAndPasswordHandler(email, password, name) {
    try {
      setLoading(true);
      const userCredential = await createUserWithEmailAndPassword(
        firebaseAuth,
        email,
        password
      );
      const user = userCredential.user;
      await updateProfile(user, { displayName: name });
      await sendEmailVerification(userCredential.user);
    } catch (error) {
      console.log(error);
      setError(error.code);
    } finally {
      setLoading(false);
    }
  }

  // Sign in the user with email and password
  async function signInWithEmailAndPasswordHandler(email, password) {
    try {
      setLoading(true);
      const userCredential = await signInWithEmailAndPassword(
        firebaseAuth,
        email,
        password
      );
      const additionalUserInfo = getAdditionalUserInfo(userCredential);

      // Create a user profile on the server for the new user
      if (additionalUserInfo.isNewUser) {
        setLoading(isMutating);
        await createUser({ arg: userCredential.user });

        if (createUserError) {
          console.log(createUserError);
          setError(createUserError);
        }
      }

      // Save the session in the cookie
      // Get the user role and subscription from the token
      // Save the user uid, rolebscription in the session
      const userUid = userCredential?.user?.uid;
      const tokenResult = await userCredential?.user?.getIdTokenResult(true);
      const role = tokenResult?.claims?.role;
      const subscription = tokenResult?.claims?.subscription;
      const sessionData = {
        uid: userUid,
        role: role,
        subscription: subscription,
      };
      const stringifiedSessionData = JSON.stringify(sessionData);

      if (stringifiedSessionData) {
        await createSession(stringifiedSessionData);
      }

      setAuthUser(formatAuthUser(userCredential.user));
    } catch (error) {
      const errorCode = error.code;
      // console.log(errorCode);
      setError(errorCode);
    } finally {
      setLoading(false);
    }
  }

  // Send a password reset email
  async function sendPasswordResetEmailHandler(email) {
    try {
      setLoading(true);
      await sendPasswordResetEmail(firebaseAuth, email);
    } catch (error) {
      console.log(error);
      setError(error);
    } finally {
      setLoading(false);
    }
  }

  // Sign in the user with Google
  async function signInWithGoogleHandler() {
    try {
      setLoading(true);
      const provider = new GoogleAuthProvider();
      const userCredential = await signInWithPopup(firebaseAuth, provider);

      const additionalUserInfo = getAdditionalUserInfo(userCredential);
      // Create a user profile for the new user
      if (additionalUserInfo.isNewUser) {
        setLoading(isMutating);
        await createUser({ arg: userCredential.user });

        if (createUserError) {
          throw createUserError;
        }
      }

      // Save the session in the cookie
      // Get the user role and subscription from the token
      // Save the user uid, rolebscription in the session
      const userUid = userCredential?.user?.uid;
      const tokenResult = await userCredential?.user?.getIdTokenResult(true);
      const role = tokenResult?.claims?.role;
      const subscription = tokenResult?.claims?.subscription;
      const sessionData = {
        uid: userUid,
        role: role,
        subscription: subscription,
      };
      const stringifiedSessionData = JSON.stringify(sessionData);

      if (stringifiedSessionData) {
        await createSession(stringifiedSessionData);
      }

      setAuthUser(formatAuthUser(userCredential.user));
    } catch (error) {
      console.log(error);
      setError(error);
    } finally {
      setLoading(false);
    }
  }

  // Sign out the user
  async function signOutHandler() {
    try {
      setLoading(true);
      await deleteSession(); //cookie
      await signOut(firebaseAuth);
      setAuthUser(null);
    } catch (error) {
      console.log(error);
      setError(error);
    } finally {
      setLoading(false);
    }
  }

  // Listen for auth state changes
  useEffect(() => {
    const unsubscribe = onAuthStateChanged(firebaseAuth, authStateChanged);
    return () => unsubscribe();
  }, []); // Developer Note: Removing this empty array will cause the effect to run infinitely and crash the app

  return {
    authUser,
    loading,
    error,
    createUserWithEmailAndPasswordHandler,
    signInWithEmailAndPasswordHandler,
    sendPasswordResetEmailHandler,
    signInWithGoogleHandler,
    signOutHandler,
  };
}
